﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.Windows.Forms;
using Microsoft.Win32;

using System.IO;
using FormSerialisation;

using Ivi.ConfigServer.Interop;

namespace PowerAmpDemoNS
{
    public partial class PowerAmpDemoGUI : Form
    {
        private PowerAmpDemoProgram TestProgram = new PowerAmpDemoProgram();
        private TestPlan_SFP myTestplan_SFP;
        private HelpViewer myHelpViewer;
        private int timesToRun = 0;
        private bool abortTest = false;
        private string lastDataLog;

        public PowerAmpDemoGUI()
        {
            InitializeComponent();
            // GUI Setup
            closeButton.Enabled = false;
            runSelectedTestsButton.Enabled = false;
            LocalxApp.Enabled = false;
            localM9391.Enabled = false;
            if (Application.StartupPath.Contains("x86")) 
                this.Text = "Agilent PXI Power Amp Demo Program (32-Bit)";
            else
                this.Text = "Agilent PXI Power Amp Demo Program (64-Bit)";
            TestProgram.configLists(testListBox, measListBox, xAppListBox);

            // Check to see if the Vsa Resource is a valid IVI Configuration Store Alias for one of the supports VSAs
            var store = new IviConfigStore();
            store.Deserialize(store.MasterLocation);
            bool validAlias = false;
            foreach (IviDriverSession driverSession in store.DriverSessions)
            {
                if (driverSession.SoftwareModuleName.Equals("AgM9393",
                                                            StringComparison.CurrentCultureIgnoreCase))
                {
                    validAlias = true;
                    // Add to the list
                    vsaResourceCBox.Items.Add(driverSession.Name);
                }
                else if (driverSession.SoftwareModuleName.Equals("AgM9391",
                                                            StringComparison.CurrentCultureIgnoreCase))
                {
                    validAlias = true;
                    // Add to the list
                    vsaResourceCBox.Items.Add(driverSession.Name);
                }
            }
            if (!validAlias)
            {
                MessageBox.Show("No VSA IVI Configuration Store Alias found for M9391A or M9393A\n" +
                    "Use VSA SFPs to create Saved Connection");
            }

            // Load from last state
            loadGuiState(Application.StartupPath + @"\lastState.xml");
            // Setup the Tool tip Help
            configureHelp();


        }

        #region Open and Close Instruments

        private void initButton_Click(object sender, EventArgs e)
        {
            Cursor.Current = Cursors.WaitCursor;
            //string selectedVsaResource = vsaResourceStringBox.Text;
            string selectedVsaResource = vsaResourceCBox.SelectedItem.ToString();

            try
            {
                // Check to see if the Vsa Resource is a valid IVI Configuration Store Alias for one of the supports VSAs
                var store = new IviConfigStore();
                store.Deserialize(store.MasterLocation);
                bool validAlias = false;
                foreach (IviDriverSession driverSession in store.DriverSessions)
                {
                    if (driverSession.SoftwareModuleName.Equals("AgM9393",
                                                                StringComparison.CurrentCultureIgnoreCase) &&
                        driverSession.Name.Equals(selectedVsaResource, StringComparison.CurrentCultureIgnoreCase))
                    {
                        validAlias = true;
                        // Create the Vsag 
                        TestProgram.createVsag("M9393A");
                        break;
                    }
                    else if (driverSession.SoftwareModuleName.Equals("AgM9391",
                                                                StringComparison.CurrentCultureIgnoreCase) &&
                        driverSession.Name.Equals(selectedVsaResource, StringComparison.CurrentCultureIgnoreCase))
                    {
                        validAlias = true;
                        // Create the Vsag 
                        TestProgram.createVsag("M9391A");
                        break;
                    }
                }
                if (!validAlias)
                {
                    Exception ex = new Exception("VSA Resource must be IVI Configuration Store Alias for M9391A or M9393A\n" +
                        "Use VSA SFPs to create Saved Connection");
                    throw (ex);
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                return;
            }

            // Initialize the Calibration Data
            if (calFileBox.Text != "")
            {
                // Cal file needs to be loaded
                FileInfo fi = new FileInfo(calFileBox.Text);
                TestProgram.vsag.readCalData(fi.DirectoryName + "\\", fi.Name);
            }
            else
            {
                TestProgram.vsag.initCalData((double)inputLossUpDown.Value, (double)outputLossUpDown.Value);
            }

            // Send GUI Instrument settings down to test library
            TestProgram.vsag.VsgResource = vsgResourceStringBox.Text;
            TestProgram.vsag.VsaResource = selectedVsaResource;
            TestProgram.vsag.M9018resource = chassisResourceStringBox.Text;
            TestProgram.vsag.simulateHardware = simulateBox.Checked;
            TestProgram.vsag.useM9300 = useM9000Box.Checked;
            TestProgram.vsag.showM90xaDisplay = showXappBox.Checked;
            TestProgram.etArb.arbResource = arbResourceBox.Text;
            TestProgram.etArb.useEtArb = etArbBox.Checked;
            TestProgram.etArb.useM9330 = m9330Button.Checked;
            TestProgram.etArb.use33522B = e33522bButton.Checked;
            TestProgram.dcSmu.useDc = dcSmuBox.Checked;
            TestProgram.dcSmu.dcResource = dcResourceStringBox.Text;
            TestProgram.dcSmu.useN6700 = n6700Button.Checked;
            TestProgram.dcSmu.useAm4xx = pxiSmuButton.Checked;
            TestProgram.dio.enableRffe = rffeBox.Checked;
            TestProgram.dio.rffeResource = rffeResourceBox.Text;

            // Initiailize the Instruments
            testStatusLabel.Text = "Initializing Instruments";
            string[] selectedXapps = xAppListBox.SelectedItems.OfType<object>()
                                                     .Select(x => x.ToString()).ToArray();
            try
            {
                TestProgram.initializeInstruments(selectedXapps);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
            // Setup the GUI to run tests
            testStatusLabel.Text = "Initialized";
            closeButton.Enabled = true;
            runSelectedTestsButton.Enabled = true;
            initButton.Enabled = false;
            simulateBox.Enabled = false;
            useM9000Box.Enabled = false;
            localM9391.Enabled = true;
            if (TestProgram.vsag.useM9300)
                LocalxApp.Enabled = true;
            testStatusLabel.Text = "Initialization Complete";
            Cursor.Current = Cursors.Default;
            tabControl1.SelectedIndex = 1;

            // Bring the form to top after initialization is complete
            this.TopMost = true;
            Application.DoEvents();
            this.TopMost = false;
        }

        private void closeButton_Click(object sender, EventArgs e)
        {
            testStatusLabel.Text = "Closing Hardware";
            TestProgram.closeInstruments();

            // Put GUI back to Init state
            testStatusLabel.Text = "Press Init to Start";
            closeButton.Enabled = false;
            runSelectedTestsButton.Enabled = false;
            initButton.Enabled = true;
            simulateBox.Enabled = true;
            useM9000Box.Enabled = true;
            LocalxApp.Checked = false;
            localM9391.Enabled = false;
            tabControl1.SelectedIndex = 0;
        }

        private void AgilentPxiDemo_FormClosing(object sender, FormClosingEventArgs e)
        {
            try
            {
                // Make sure instruments are closed before the GUI exits
                TestProgram.closeInstruments();
                // Put GUI back to Init state
                testStatusLabel.Text = "Press Init to Start";
                closeButton.Enabled = false;
                runSelectedTestsButton.Enabled = false;
                initButton.Enabled = true;
                simulateBox.Enabled = true;
                useM9000Box.Enabled = true;
                LocalxApp.Checked = false;
                FormSerialisor.Serialise(this, Application.StartupPath + @"\lastState.xml");
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        //protected override void OnClosing(System.ComponentModel.CancelEventArgs e)
        //{
        //     workaround for problem when closing a component of the M90Xa
        //    Control.CheckForIllegalCrossThreadCalls = false;

        //    base.OnClosing(e);
        //}
        #endregion

        #region Configure Test Conditions

        private void runCalButton_Click(object sender, EventArgs e)
        {
            openFileDialog1.Filter = "csv files (*.csv)|*.csv|All files (*.*)|*.*";
            openFileDialog1.CheckFileExists = false;
            openFileDialog1.AddExtension = true;
            openFileDialog1.FileName = "MyCalData";
            openFileDialog1.ShowDialog();
            try
            {
                TestProgram.runPowerCalibration(sensorResourceBox.Text, openFileDialog1.FileName);
                FileInfo fi = new FileInfo(openFileDialog1.FileName);
                TestProgram.vsag.readCalData(fi.DirectoryName + "\\", fi.Name);
                calFileBox.Text = openFileDialog1.FileName;
                System.Diagnostics.Process.Start(openFileDialog1.FileName);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }
        
        private void loadCalButton_Click(object sender, EventArgs e)
        {
            openFileDialog1.Filter = "csv files (*.csv)|*.csv|All files (*.*)|*.*";
            openFileDialog1.ShowDialog();
            calFileBox.Text = openFileDialog1.FileName;
            try
            {
                FileInfo fi = new FileInfo(openFileDialog1.FileName);
                TestProgram.vsag.readCalData(fi.DirectoryName + "\\", fi.Name);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void loadNominalCalButton_Click(object sender, EventArgs e)
        {
            TestProgram.vsag.initCalData((double)inputLossUpDown.Value, (double)outputLossUpDown.Value);
            calFileBox.Text = "";
        }

        private void TestProgram_ValueChanged(object sender, EventArgs e)
        {
            setTestProgramValues();
            TestProgram.setLteSetupDirty();
            TestProgram.setLteTddSetupDirty();
        }

        private void setTestProgramValues()
        {
            // Servo Loop Values
            TestProgram.targetPout = (double)pOutUpDown.Value;
            TestProgram.targetGain = (double)gainUpDown.Value;
            TestProgram.poutMargin = (double)toleranceUpDown.Value;

            // WLAN Pulse Settings
            TestProgram.dutyCycle = (double)wlanDutyCycle.Value/100;
            TestProgram.dcLeadTime = (double)dcLeadTime_us.Value * 1e-6;
            TestProgram.dcLagTime = (double)dcLagTime_us.Value * 1e-6;
            TestProgram.pulseVoltage = (double)pulseVoltageControl.Value;
            TestProgram.etArb.arbTrigDelay = (double)dEvmDelay.Value / 1e9;
            TestProgram.useDynamicEvm = dynamicEvmBox.Checked;
            TestProgram.evmAvgCount = (int) evmAverages.Value;
            TestProgram.semAvgCount = (int)semAverages.Value;

            // LTE ET Settings
            TestProgram.iqDelayOffset = (double)iqDelayControl.Value * 1e-9;
            TestProgram.arbAmplitude = (double)amplitudeVoltageControl.Value;
            TestProgram.arbOffset = (double)offsetVoltageControl.Value;
            TestProgram.enableEt = enableEtBox.Checked;
        }

        private void freqDefaultsButton_Click(object sender, EventArgs e)
        {
            TestProgram.initTestFreqs();
            testFreqFileBox.Text = "";
            fillTestFreqDisplay();
        }

        private void loadTestFreqsButton_Click(object sender, EventArgs e)
        {
            openFileDialog1.Filter = "csv files (*.csv)|*.csv|All files (*.*)|*.*";
            openFileDialog1.ShowDialog();
            testFreqFileBox.Text = openFileDialog1.FileName;
            try
            {
                FileInfo fi = new FileInfo(openFileDialog1.FileName);
                TestProgram.readFreqData(fi.DirectoryName + "\\", fi.Name);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
            fillTestFreqDisplay();
        }

        private void fillTestFreqDisplay()
        {
            testFreqGrid.Rows.Clear();
            int numRows = Math.Max(TestProgram.cellFreqs.Length, TestProgram.wlanFreqs.Length);
            for (int i = 0; i < numRows; i++)
            {
                string[] cellText = { "", "" };
                if (i < TestProgram.cellFreqs.Length)
                    cellText[0] = (TestProgram.cellFreqs[i]).ToString("N0");
                if (i < TestProgram.wlanFreqs.Length)
                    cellText[1] = (TestProgram.wlanFreqs[i]).ToString("N0");
                testFreqGrid.Rows.Add(cellText);
            }
        }

        private void loadDcSetupButton_Click(object sender, EventArgs e)
        {
            openFileDialog1.Filter = "csv files (*.csv)|*.csv|All files (*.*)|*.*";
            openFileDialog1.ShowDialog();
            dcSetupFileBox.Text = openFileDialog1.FileName;
            try
            {
                FileInfo fi = new FileInfo(openFileDialog1.FileName);
                TestProgram.dcSmu.readDcSetupData(fi.DirectoryName + "\\", fi.Name);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
            fillDcSetupDisplay();
        }

        private void fillDcSetupDisplay()
        {
            dcSetupGrid.Rows.Clear();
            int numRows = TestProgram.dcSmu.dcVoltage.Length;
            for (int i = 0; i < numRows; i++)
            {
                string[] cellText = { "", "No", "", "", "No" };
                cellText[0] = (i+1).ToString();
                if (TestProgram.dcSmu.enableDcChan[i]) cellText[1] = "Yes";
                cellText[2] = TestProgram.dcSmu.dcVoltage[i].ToString("0.000");
                cellText[3] = TestProgram.dcSmu.dcCurrLim[i].ToString("0.000");
                if (TestProgram.dcSmu.doCurrMeas[i]) cellText[4] = "Yes";
                dcSetupGrid.Rows.Add(cellText);
            }
        }

        private void loadRffeButton_Click(object sender, EventArgs e)
        {
            openFileDialog1.Filter = "csv files (*.csv)|*.csv|All files (*.*)|*.*";
            openFileDialog1.ShowDialog();
            rffeCommandFileBox.Text = openFileDialog1.FileName;
            try
            {
                FileInfo fi = new FileInfo(openFileDialog1.FileName);
                TestProgram.dio.readRffeCommandData(fi.DirectoryName + "\\", fi.Name);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
            fillRffeDisplay();
        }

        private void fillRffeDisplay(bool showReadDAta = false)
        {
            if (TestProgram.dio.rffeAddress == null) return;
            rffeGridView.Rows.Clear();
            int numRows = TestProgram.dio.rffeAddress.Length;
            for (int i = 0; i < numRows; i++)
            {
                string[] cellText = { "", "", "", "No", "---" };
                cellText[0] = TestProgram.dio.rffeAddress[i].ToString();
                cellText[1] = TestProgram.dio.rffeRegister[i].ToString();
                cellText[2] = TestProgram.dio.rffeData[i].ToString();
                if (TestProgram.dio.rffeReadback[i])
                {
                    cellText[3] = "Yes";
                    if (showReadDAta)
                        cellText[4] = TestProgram.dio.rffeReadData[i].ToString();
                }
                rffeGridView.Rows.Add(cellText);
            }
        }


        private void resetGuiButton_Click(object sender, EventArgs e)
        {
            loadGuiState(Application.StartupPath + @"\defaults.xml");
        }

        private void recallStateButton_Click(object sender, EventArgs e)
        {
            openFileDialog1.Filter = "xml files (*.xml)|*.xml|All files (*.*)|*.*";
            openFileDialog1.InitialDirectory = Application.StartupPath;
            openFileDialog1.ShowDialog();
            testFreqFileBox.Text = openFileDialog1.FileName;
            loadGuiState(openFileDialog1.FileName);
        }
        private void loadGuiState(string fileName)
        {
            try
            {
                FormSerialisor.Deserialise(this, fileName);
                if (testFreqFileBox.Text != "")
                {
                    FileInfo fi = new FileInfo(testFreqFileBox.Text);
                    try
                    {

                        TestProgram.readFreqData(fi.DirectoryName + "\\", fi.Name);
                    }
                    catch
                        {
                            MessageBox.Show("Error Loading Frequency data from:\n" + fi.FullName + "\nLoading default Frequency Values.");
                            TestProgram.initTestFreqs();
                            testFreqFileBox.Text = "";
                        }

                }
                else
                {
                    TestProgram.initTestFreqs();
                }
                fillTestFreqDisplay();

                if (dcSetupFileBox.Text != "")
                {
                    FileInfo fi = new FileInfo(dcSetupFileBox.Text);
                    try
                    {

                        TestProgram.dcSmu.readDcSetupData(fi.DirectoryName + "\\", fi.Name);
                        fillDcSetupDisplay();
                    }
                    catch
                    {
                        MessageBox.Show("Error Loading DC setup data from:\n" + fi.FullName);
                        dcSetupFileBox.Text = "";
                    }

                }
                if (rffeCommandFileBox.Text != "")
                {
                    FileInfo fi = new FileInfo(rffeCommandFileBox.Text);
                    try
                    {

                        TestProgram.dio.readRffeCommandData(fi.DirectoryName + "\\", fi.Name);
                        fillRffeDisplay();
                    }
                    catch
                    {
                        MessageBox.Show("Error Loading RFFE Command data from:\n" + fi.FullName);
                        rffeCommandFileBox.Text = "";
                    }

                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
                MessageBox.Show("Error Loading Saved State from:\n" + fileName + "\nCheck GUI Settings.");
            }
        }

        private void saveStateButton_Click(object sender, EventArgs e)
        {
            saveFileDialog1.Filter = "xml files (*.xml)|*.xml|All files (*.*)|*.*";
            saveFileDialog1.InitialDirectory = Application.StartupPath;
            saveFileDialog1.ShowDialog();
            FormSerialisor.Serialise(this, saveFileDialog1.FileName);
        }
        #endregion

        #region Run Tests and Local Instruments
        private void runSelectedTests_Click(object sender, EventArgs e)
        {
            Cursor.Current = Cursors.WaitCursor;

            TestProgram.vsag.fftServo = fftServoButton.Checked;
            TestProgram.vsag.fftAcpr = fftAcprButton.Checked;
            
            runSelectedTestsButton.Enabled = false;
            tabControl1.SelectedIndex = 2;
            abortTest = false;
            LocalxApp.Enabled = false;
            localM9391.Enabled = false;
            setTestProgramValues();
            TestProgram.dataLog.loggingData = logDataBox.Checked;
            TestProgram.dataLog.updateResultWindow = showResultsBox.Checked;
            if (TestProgram.dataLog.loggingData)
            {
                lastDataLog = "logFile" + DateTime.Now.ToString("yyyyMMddHHmmss") + ".csv";
                TestProgram.dataLog.openDataLog("C:\\Temp\\", lastDataLog);
            }

            testStatusLabel.Text = "Running Selected Tests " + repeatCount.Value.ToString() + " Times...";
            Cursor.Current = Cursors.WaitCursor;

            TestProgram.dataLog.createHeader((int)repeatCount.Value);
            
            // Setup for Power Search and VSG Alignment
            TestProgram.vsag.initVsagAlignments();
            TestProgram.vsag.alignVsa = alignVsaBox.Checked;
            TestProgram.vsag.alignVsg = alignVsgBox.Checked;

            resultsBox.Text = "";
            TestProgram.dataLog.resultsString = "";

            timesToRun = (int)repeatCount.Value;

            var oldMode = System.Runtime.GCSettings.LatencyMode;
            int dutsRun = 0;

            string[] selectedTests = testListBox.SelectedItems.OfType<object>()
                                          .Select(x => x.ToString()).ToArray();
            string[] selectedMeas = measListBox.SelectedItems.OfType<object>()
                                          .Select(x => x.ToString()).ToArray();

            try
            {
                // Turn on the DC Supplies
                TestProgram.dcSmu.setupSmus();
                TestProgram.dcSmu.closeSmuOutputRelays();
                // Send the RFFE Commands
                TestProgram.dio.sendRffeCommands();
                // Update the RFFE Display to show any readback data
                fillRffeDisplay(true);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }

            // Loop over the selected tests
            do
            {
                resultsBox.Text = "";
                try
                {
                    // If the poutIncement control has a value != 0, the target power
                    // will change for each repeat of the DUT, allowing the program to 
                    // sweep power as well as frequency
                    TestProgram.targetPout = (double)pOutUpDown.Value + (double)poutIncement.Value * dutsRun;
                    TestProgram.runSelectedTests(selectedTests, selectedMeas);
                }
                catch (Exception ex)
                {
                    MessageBox.Show(ex.Message);
                }
                dutsRun++;
                testStatusLabel.Text = "Running Selected Test Number " + (dutsRun).ToString();
                resultsBox.Text = TestProgram.dataLog.resultsString;
                TestProgram.dataLog.resultsString = "";
                Application.DoEvents();
                //System.Runtime.GCSettings.LatencyMode = oldMode;
                System.Threading.Thread.Sleep(1);
            } while (dutsRun < timesToRun && !abortTest);
            if (TestProgram.dataLog.loggingData) TestProgram.dataLog.closeDataLog();
            if (dcOffBox.Checked) TestProgram.dcSmu.openSmuOutputRelays();

            Cursor.Current = Cursors.Default;
            testStatusLabel.Text = "Running Selected Test " + dutsRun.ToString() + " Times Complete!";
            LocalxApp.Enabled = true;
            localM9391.Enabled = true;
            runSelectedTestsButton.Enabled = true;
            Cursor.Current = Cursors.Default;
        }

        private void abortButton_Click(object sender, EventArgs e)
        {
            abortTest = true;
        }

        private void showXappBox_CheckedChanged(object sender, EventArgs e)
        {
            if (TestProgram.vsag != null)
                TestProgram.vsag.setXAppVisible(showXappBox.Checked);
        }
        private void clearResults_Click(object sender, EventArgs e)
        {
            resultsBox.Text = "";
            TestProgram.dataLog.resultsString = "";
        }

        private void LocalxApp_CheckedChanged(object sender, EventArgs e)
        {
            if (TestProgram.vsag == null) return;
            TestProgram.vsag.setXAppMode(LocalxApp.Checked);
            if (LocalxApp.Checked == true)
            {
                runSelectedTestsButton.Enabled = false;
                showXappBox.Enabled = false;
                localM9391.Enabled = false;
                TestProgram.vsag.showVsgMonitor();
            }
            else
            {
                runSelectedTestsButton.Enabled = true;
                showXappBox.Enabled = true;
                localM9391.Enabled = true;
                TestProgram.vsag.stopVsgMonitor();
             }
        }

        private void localM9391_CheckedChanged(object sender, EventArgs e)
        {
            if (TestProgram.vsag == null)
            {
                localM9391.Checked = false;
                return;
            }
            if (localM9391.Checked == true)
            {
                runSelectedTestsButton.Enabled = false;
                LocalxApp.Enabled = false;
                TestProgram.vsag.showVsgMonitor();
                TestProgram.vsag.showVsaMonitor();
            }
            else
            {
                runSelectedTestsButton.Enabled = true;
                LocalxApp.Enabled = true;
                TestProgram.vsag.stopVsgMonitor();
                TestProgram.vsag.stopVsaMonitor();
            }
        }

        private void debugTestBox_CheckedChanged(object sender, EventArgs e)
        {
            if (debugTestBox.Checked == true)
            {
                runSelectedTestsButton.Enabled = false;
                LocalxApp.Enabled = false;
                myTestplan_SFP = new TestPlan_SFP(TestProgram);
                myTestplan_SFP.Show();
            }
            else
            {
                runSelectedTestsButton.Enabled = true;
                LocalxApp.Enabled = true;
                myTestplan_SFP.Dispose();
                myTestplan_SFP = null;
            }
        }

        private void openDataLogButton_Click(object sender, EventArgs e)
        {
            if (lastDataLog == null)
                MessageBox.Show("No Data Log File has been Generated");
            else
            {
                string fn = @"C:\Temp\" + lastDataLog;
                FileInfo fi = new FileInfo(fn);
                if (fi.Exists)
                    System.Diagnostics.Process.Start(fn);
                else
                    MessageBox.Show("Data Log File does not exist");
            }
        }

        private void copyTextbutton_Click(object sender, EventArgs e)
        {
            System.Windows.Forms.Clipboard.SetText(resultsBox.Text);
        }

        #endregion

        #region Help Setup
        private void viewHelpButton_Click(object sender, EventArgs e)
        {
            string helpFilePath = Application.StartupPath + @"\Agilent PXI Power Amp Demo Program User Guide.pdf";
            myHelpViewer = new HelpViewer(helpFilePath);
            myHelpViewer.Show();
        }

        private void configureHelp()
        {
            // Create a Tool Tip to show "Hover help" for controls
            ToolTip toolTip1 = new ToolTip();

            // Set up the delays for the ToolTip.
            toolTip1.AutoPopDelay = 5000;
            toolTip1.InitialDelay = 1000;
            toolTip1.ReshowDelay = 500;
            // Force the ToolTip text to be displayed whether or not the form is active.
            toolTip1.ShowAlways = true;
            string helpText = "";

            // Instrument Setup Tab
            helpText = "Check to use the Instruments in simulation Mode";
            toolTip1.SetToolTip(this.simulateBox, helpText);

            helpText = "VISA Resource for M9381A VSG.  Can be a list of Addresses for the Modules\nor a Saved Configuration from the M9381A SFP";
            toolTip1.SetToolTip(this.vsgResourceStringBox, helpText);
            toolTip1.SetToolTip(this.label1, helpText);

            helpText = "VISA Resource for M9391A VSG.  Can be a list of Addresses for the Modules\nor a Saved Configuration from the M9391A SFP.\nWhen using X-Apps, this must be a Saved Configuration Name";
            toolTip1.SetToolTip(this.vsaResourceStringBox, helpText);
            toolTip1.SetToolTip(this.label2, helpText);

            helpText = "Check to use the M90XA X-Apps Software.\nX-App Software requried for EVM and SEM measurements.";
            toolTip1.SetToolTip(this.useM9000Box, helpText);

            helpText = "Select the X-Apps to be preloaded on initialization";
            toolTip1.SetToolTip(this.xAppListBox, helpText);

            // DUT Setup Tab
            // RF Settings Tab
            helpText = "Target Output Power for Servo Loops during Device Tests.\nEach Test will adjust DUT ouptut power to this level before running tests.";
            toolTip1.SetToolTip(this.pOutUpDown, helpText);

            helpText = "Expected Gain of the DUT at the Target Output Power.\nUsed to set the starting output power of the VSG for the Servo Loops.";
            toolTip1.SetToolTip(this.gainUpDown, helpText);

            helpText = "Tolerance for Output Power for Servo Loops during Device Tests.\nServo Loop will Adjust DUT Output Power to Target Power +/- Tolerance.";
            toolTip1.SetToolTip(this.toleranceUpDown, helpText);

            helpText = "Target Power Increment for Repeated Tests.\nSet to > 0 and set Repeat Count to > 0 to run power sweep.";
            toolTip1.SetToolTip(this.poutIncement, helpText);

            helpText = "Frequencies that will be Tested in DUT Tests.\nCellular Standards will use the Frequencies in the Cellular Column\nWLAN Standards will use the Frequencies in the WLAN Column.";
            toolTip1.SetToolTip(this.testListBox, helpText);

            helpText = "Reset the Test Frequencies to the Default Values.";
            toolTip1.SetToolTip(this.freqDefaultsButton, helpText);

            helpText = "Load the Test Frequencies from a .CSV file";
            toolTip1.SetToolTip(this.testFreqFileBox, helpText);
            toolTip1.SetToolTip(this.loadTestFreqsButton, helpText);

            helpText = "Set Input and Ouptut Calibration values that will be used for all test frequencies.\n  Press Load after changing values";
            toolTip1.SetToolTip(this.inputLossUpDown, helpText);
            toolTip1.SetToolTip(this.outputLossUpDown, helpText);
            toolTip1.SetToolTip(this.loadNominalCalButton, helpText);

            helpText = "Load the Calibration Values from a .CSV file.\nAllows for frequency dependent calibration data.";
            toolTip1.SetToolTip(this.calFileBox, helpText);
            toolTip1.SetToolTip(this.loadCalButton, helpText);

            helpText = "Run the Calibration Procedure using Power Sensor for Current Frequencies.\n Save data to a .CSV file.\nLoad Calibration data after procedure.";
            toolTip1.SetToolTip(this.runCalButton, helpText);

            // DC Settings Tab

            // WLAN Settings Tab
            helpText = "Set the Duty Cycle of the WLAN Signal from the VSG";
            toolTip1.SetToolTip(this.wlanDutyCycle, helpText);

            helpText = "Enable generating the PA Enable Signal for Dynamic EVM";
            toolTip1.SetToolTip(this.dynamicEvmBox, helpText);

            helpText = "Amplitude of the PA Enable Signal into 50 ohm load";
            toolTip1.SetToolTip(this.pulseVoltageControl, helpText);

            helpText = "Nominal Trigger Delay to align the RF and PA Enable Pulses";
            toolTip1.SetToolTip(this.dEvmDelay, helpText);

            helpText = "Time for PA Enable Pulse Leading Edge to Lead the RF Burst";
            toolTip1.SetToolTip(this.dcLeadTime_us, helpText);

            helpText = "Time for PA Enable Pulse Trailing Edge to Trail the RF Burst";
            toolTip1.SetToolTip(this.dcLagTime_us, helpText);

            //LTE ET Settings Tab
            helpText = "Enable generating the RF Envelope Signal";
            toolTip1.SetToolTip(this.enableEtBox, helpText);

            helpText = "Amplitude of the the RF Envelope Signal into 50 ohm load";
            toolTip1.SetToolTip(this.amplitudeVoltageControl, helpText);

            helpText = "DC Offset of the the RF Envelope Signal into 50 ohm load";
            toolTip1.SetToolTip(this.offsetVoltageControl, helpText);

            helpText = "Adjustment for the RF and Envelope Tracking Delay, from nominal value for each bandwidth";
            toolTip1.SetToolTip(this.iqDelayControl, helpText);

            // Results Tab
            helpText = "Check to Display the detailed test results.\nMay increase test time.\nDisable to get best test times.";
            toolTip1.SetToolTip(this.showResultsBox, helpText);

            helpText = "Clear all of the data in the Test Results Window.";
            toolTip1.SetToolTip(this.clearResultsButton, helpText);

            helpText = "Copy all of the data in the Test Results Window to the Windows Clipboard.";
            toolTip1.SetToolTip(this.copyTextbutton, helpText);

            // Main Form
            helpText = "Intialize the instrument drivers and X-Apps and load all waveforms into VSG.";
            toolTip1.SetToolTip(this.initButton, helpText);

            helpText = "Close the instruments drivers and X-Apps.";
            toolTip1.SetToolTip(this.closeButton, helpText);

            helpText = "Check to Display the X-Apps User Interface.\nMay increase test time.\nDisable to get best test times.";
            toolTip1.SetToolTip(this.showXappBox, helpText);

            helpText = "Check to Allow Local Control the X-Apps and VSG.\nOnly Available when tests are not running.\nUncheck to end local control.";
            toolTip1.SetToolTip(this.LocalxApp, helpText);

            helpText = "Check to Allow Local Control the VSA (IVI Driver) and VSG.\nOnly Available when tests are not running.\nUncheck to end local control";
            toolTip1.SetToolTip(this.localM9391, helpText);

            helpText = "Select the Standards to be tested on the DUT when Run Selected Tests in pressed";
            toolTip1.SetToolTip(this.testListBox, helpText);

            helpText = "Select the Measurements to be tested on the DUT when Run Selected Tests in pressed.\nMeasurements will be run on each standard, if they are supported for that standard.\nX-Apps must be running to measure EVM and SEM";
            toolTip1.SetToolTip(this.measListBox, helpText);

            helpText = "Run the Measurements.\nMeasurements will be run on each standard, if they are supported for that standard.\n";
            toolTip1.SetToolTip(this.runSelectedTestsButton, helpText);

            helpText = "Number of times the DUT tests will run when the Run Selected Tests button is pressed.\n";
            toolTip1.SetToolTip(this.repeatCount, helpText);

            helpText = "Check to save the measured data to the CSV file C:\\Temp\\logfile.csv.\nLog file includes results for each repetition and repeatability statistics";
            toolTip1.SetToolTip(this.logDataBox, helpText);

            helpText = "Press to abort tests at end of current repetition.";
            toolTip1.SetToolTip(this.abortButton, helpText);

            helpText = "Save the Current State of the User Interface to a File that can be recalled later";
            toolTip1.SetToolTip(this.saveStateButton, helpText);

            helpText = "Recall the State of the User Interface from a File that was previoulsy saved";
            toolTip1.SetToolTip(this.recallStateButton, helpText);

            helpText = "Reset the State of the User Interface to the Default Values";
            toolTip1.SetToolTip(this.resetGuiButton, helpText);

            helpText = "Open the most recent data log in the default application for .csv files";
            toolTip1.SetToolTip(this.openDataLogButton, helpText);

            helpText = "Show the Users Guide";
            toolTip1.SetToolTip(this.viewHelpButton, helpText);

        }
        #endregion

     }
}
